// rs_script.js
// Reed–Solomon ファイル誤り訂正ツール（複数ファイル対応＋整合性検証＋グラフ表示）

(async function(){
  const BLOCK_SIZE = 255;
  const logEl = document.getElementById('log');
  const progressEl = document.getElementById('progress');
  const rateEl = document.getElementById('ecRate');
  const rateLabel = document.getElementById('ecRateLabel');
  const chartEl = document.getElementById('chart');
  let chart;

  function log(...msg){
    logEl.textContent += msg.join(' ') + '\n';
    logEl.scrollTop = logEl.scrollHeight;
  }
  function setProgress(p){ progressEl.value = p; }

  const strToU8 = s => new TextEncoder().encode(s);
  const u8ToStr = u => new TextDecoder().decode(u);
  const readU32BE = (u, o=0)=> (u[o]<<24)|(u[o+1]<<16)|(u[o+2]<<8)|u[o+3];
  const writeU32BE = n => new Uint8Array([(n>>24)&255,(n>>16)&255,(n>>8)&255,n&255]);

  async function calcSHA256(u8){
    const hashBuffer = await crypto.subtle.digest("SHA-256", u8);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map(b=>b.toString(16).padStart(2,"0")).join("");
  }

  rateEl.addEventListener('input', ()=> rateLabel.textContent = `${rateEl.value}%`);
  function readFile(file){
    return new Promise((res,rej)=>{
      const fr=new FileReader();
      fr.onload=()=>res(fr.result);
      fr.onerror=()=>rej(fr.error);
      fr.readAsArrayBuffer(file);
    });
  }

  function drawChart(successRate){
    if(chart) chart.destroy();
    chart = new Chart(chartEl, {
      type:'doughnut',
      data:{
        labels:['成功ブロック','失敗ブロック'],
        datasets:[{data:[successRate,100-successRate],
          backgroundColor:['#2ca66c','#e55353']}]
      },
      options:{
        plugins:{legend:{position:'bottom'}},
        cutout:'65%',
        animation:{duration:800}
      }
    });
  }

  // Encode (multiple)
  document.getElementById('encodeBtn').addEventListener('click', async()=>{
    const files = Array.from(document.getElementById('encodeFiles').files||[]);
    if(files.length===0){alert('ファイルを選択してください。');return;}
    const rate = Number(rateEl.value)/100;
    const ecBytes = Math.max(2, Math.min(254, Math.floor(BLOCK_SIZE * rate / (1+rate))));
    const dataPerBlock = BLOCK_SIZE - ecBytes;
    log(`選択ファイル数：${files.length}　誤り訂正率=${rateEl.value}% (ecBytes=${ecBytes})`);
    setProgress(0);

    for(let fIndex=0; fIndex<files.length; fIndex++){
      const file = files[fIndex];
      log(`\n---[${fIndex+1}/${files.length}] ${file.name}---`);
      try{
        const ab = await readFile(file);
        const data = new Uint8Array(ab);
        const hashHex = await calcSHA256(data);
        log("SHA-256:", hashHex);

        const field = GenericGF.QR_CODE_FIELD_256();
        const encoder = new ReedSolomonEncoder(field);
        const blocks=[];
        let pos=0;
        while(pos<data.length){
          const take=Math.min(dataPerBlock,data.length-pos);
          const buf=new Int32Array(BLOCK_SIZE);
          for(let i=0;i<take;i++) buf[i]=data[pos+i];
          encoder.encode(buf,ecBytes);
          const u8=new Uint8Array(BLOCK_SIZE);
          for(let i=0;i<BLOCK_SIZE;i++) u8[i]=buf[i]&0xff;
          blocks.push(u8);
          pos+=take;
          setProgress((pos/data.length)*90);
        }
        const header={magic:"RSF1",filename:file.name,originalSize:data.length,ecBytes,blockSize:BLOCK_SIZE,dataPerBlock,hash:hashHex};
        const headerStr=JSON.stringify(header);
        const headerU8=strToU8(headerStr);
        const headerLen=writeU32BE(headerU8.length);
        const blob=new Blob([headerLen,headerU8,...blocks],{type:"application/octet-stream"});
        const a=document.createElement('a');
        a.href=URL.createObjectURL(blob);
        a.download=file.name+".rsf";
        a.click();
        URL.revokeObjectURL(a.href);
        log("符号化完了 → "+a.download);
      }catch(e){log("符号化エラー:",e.message);}
    }
    setProgress(100);
    log("\nすべてのファイルの符号付加が完了しました。");
  });

  // 予測
  document.getElementById('encodePreviewBtn').addEventListener('click',()=>{
    const files = Array.from(document.getElementById('encodeFiles').files||[]);
    if(files.length===0){alert('ファイルを選択してください。');return;}
    const rate = Number(rateEl.value)/100;
    const ecBytes = Math.max(2, Math.min(254, Math.floor(BLOCK_SIZE * rate / (1+rate))));
    const dataPerBlock = BLOCK_SIZE - ecBytes;
    let msg=`誤り訂正率: ${rateEl.value}% (ecBytes=${ecBytes})\n`;
    for(const f of files){
      const blocks=Math.ceil(f.size/dataPerBlock);
      msg+=`\n${f.name}: ${f.size} bytes → 約${blocks*BLOCK_SIZE+100} bytes`;
    }
    alert(msg);
  });

  // Decode (multiple)
  document.getElementById('decodeBtn').addEventListener('click', async()=>{
    const files = Array.from(document.getElementById('decodeFiles').files||[]);
    if(files.length===0){alert('RSFファイルを選択してください。');return;}
    let totalBlocks=0, totalFails=0;

    for(let fIndex=0; fIndex<files.length; fIndex++){
      const file = files[fIndex];
      log(`\n---[復元 ${fIndex+1}/${files.length}] ${file.name}---`);
      setProgress(0);
      try{
        const ab = await readFile(file);
        const u8 = new Uint8Array(ab);
        const headerLen = readU32BE(u8,0);
        const header = JSON.parse(u8ToStr(u8.subarray(4,4+headerLen)));
        if(header.magic!=="RSF1") throw new Error("不正なRSFファイルです。");
        const {originalSize,ecBytes,blockSize,dataPerBlock,hash}=header;
        const body=u8.subarray(4+headerLen);
        const blocks=Math.floor(body.length/blockSize);
        const field=GenericGF.QR_CODE_FIELD_256();
        const decoder=new ReedSolomonDecoder(field);
        const out=new Uint8Array(originalSize);
        let pos=0,fails=[];
        for(let i=0;i<blocks;i++){
          const block=body.subarray(i*blockSize,(i+1)*blockSize);
          const recv=new Int32Array(blockSize);
          for(let j=0;j<blockSize;j++) recv[j]=block[j];
          try{decoder.decode(recv,ecBytes);}
          catch(e){fails.push(i);}
          const take=Math.min(dataPerBlock,originalSize-pos);
          for(let k=0;k<take;k++) out[pos+k]=recv[k]&0xff;
          pos+=take;
          if(i%5===0) setProgress((i/blocks)*90);
        }
        totalBlocks+=blocks;
        totalFails+=fails.length;

        if(fails.length>0) log(`⚠ ${fails.length}ブロックの復号に失敗しました。`);
        else log("全ブロック正常に復元されました。");

        // SHA-256検証
        const newHash=await calcSHA256(out);
        log("期待値（ヘッダー）:",hash);
        log("実測値（復元後）:",newHash);
        if(newHash===hash)
          log("✅ 整合性OK：ハッシュ一致（完全復元）");
        else
          log("⚠ ハッシュ不一致：破損の可能性あり！");

        const a=document.createElement('a');
        a.href=URL.createObjectURL(new Blob([out]));
        a.download=header.filename||'restored.bin';
        a.click();
        URL.revokeObjectURL(a.href);
        setProgress(100);
        log("復元完了 → "+a.download);
      }catch(e){
        log("復元エラー:",e.message);
      }
    }
    const successRate = totalBlocks===0 ? 100 : ((totalBlocks-totalFails)/totalBlocks*100);
    log(`\n全体ブロック成功率：${successRate.toFixed(1)}%`);
    drawChart(successRate);
  });
})();
