/* script.js — v2.13.1
   保存: script.js (UTF-8)
*/

/* ---------------------------
   DOM helper
   --------------------------- */
const $ = id => document.getElementById(id);

/* ---------------------------
   定数 / 状態
   --------------------------- */
const INITIAL_DISPLAY = 10; // 初期表示枚数
const DRAW_BATCH = 10;      // 1回で描画する枚数（スクロールで追加）
const BATCH_SLEEP_MS = 2;   // バッチ間で挿入する短い待機(ms) — UI凍結防止
const PROGRESS_LOG_INTERVAL_MS = 1000; // 進捗ログを出す頻度

let qItems = [];            // 生成済みの行（HEADER + CHUNK etc.）
let generatedCount = 0;
let renderedIndex = 0;
let rendering = false;
let slideshowTimer = null;
let modalIndex = 0;
let lastProgressLogAt = 0;
let originalFileName = '';

// 復元側の状態
let restored = {
  fragments: {},
  totalChunks: 0,
  name: '',
  mime: '',
  hash: '',
  received: 0,
  isRunning: false
};

// カメラ
let cameraStream = null;
let cameraVideoEl = null;
let cameraAnim = null;

// recent dedupe map
const recentSeen = new Map();

/* ---------------------------
   基本ユーティリティ（UTF-8対応 base64 / CRC / SHA256 など）
   --------------------------- */

// 安全な Base64 変換 (ZIPなど任意バイナリ対応)
function u8ToB64(u8) {
  if (!(u8 instanceof Uint8Array)) {
    throw new Error('u8ToB64 expects Uint8Array');
  }
  let binary = '';
  const len = u8.length;
  const chunk = 0x8000; // 32Kずつ処理
  for (let i = 0; i < len; i += chunk) {
    const sub = u8.subarray(i, i + chunk);
    // String.fromCharCode() ではなく DataView 経由
    binary += Array.from(sub, b => String.fromCharCode(b)).join('');
  }
  return btoa(binary);
}

function b64ToU8(b64) {
  // ✅ Base64以外の文字（空白・改行など）を除去
  b64 = b64.replace(/[^A-Za-z0-9+/=]/g, '');

  // ✅ パディング調整（長さを4の倍数に揃える）
  while (b64.length % 4 !== 0) b64 += '=';

  try {
    const bin = atob(b64);
    const u8 = new Uint8Array(bin.length);
    for (let i = 0; i < bin.length; i++) {
      u8[i] = bin.charCodeAt(i);
    }
    return u8;
  } catch (e) {
    console.error('b64ToU8 decode error:', e, '\npreview:', b64.slice(0, 80));
    throw e;
  }
}


function strToB64Utf8(str) {
  const u8 = new TextEncoder().encode(str);
  return u8ToB64(u8);
}

// UTF-8対応のBase64→文字列デコード
function b64ToStrUtf8(b64) {
  if (!b64) return '';
  b64 = b64.replace(/[^A-Za-z0-9+/=]/g, '');
  while (b64.length % 4 !== 0) b64 += '=';
  try {
    const bin = atob(b64);
    const bytes = Uint8Array.from(bin, c => c.charCodeAt(0));
    return new TextDecoder().decode(bytes);
  } catch (e) {
    console.error('b64ToStrUtf8 decode error:', e, '\npreview:', b64.slice(0, 80));
    return '';
  }
}


async function sha256Hex(input) {
  let buf = input;
  if (input instanceof Uint8Array) buf = input.buffer;
  const h = await crypto.subtle.digest('SHA-256', buf);
  return Array.from(new Uint8Array(h)).map(b => b.toString(16).padStart(2, '0')).join('');
}

/* CRC32 table generator (fast) */
const CRC32 = (function () {
  const table = new Uint32Array(256);
  for (let i = 0; i < 256; i++) {
    let c = i;
    for (let j = 0; j < 8; j++) {
      c = (c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1);
    }
    table[i] = c >>> 0;
  }
  return function crc32(buf) {
    let crc = 0xFFFFFFFF;
    for (let i = 0; i < buf.length; i++) {
      crc = (crc >>> 8) ^ table[(crc ^ buf[i]) & 0xFF];
    }
    return (crc ^ 0xFFFFFFFF) >>> 0;
  };
})();

/* sanitize filename */
function sanitizeFilename(name) {
  if (!name) return `file_${Date.now()}`;
  // Windows・macOS・Linux で禁止されている文字のみを置換
  let safe = name.replace(/[\x00-\x1f<>:"/\\|?*]/g, '_');
  // 日本語など非ASCII文字は許可（normalizeはそのままでOK）
  try {
    safe = safe.normalize('NFC'); // 結合文字を正規化するだけ
  } catch {
    safe = 'file_' + Date.now();
  }
  return safe.slice(0, 200);
}


/* ---------------------------
   ログ表示ユーティリティ（#logAreaへプリペンド）
   --------------------------- */
function log(msg, important = false) {
  const area = $('logArea');
  const time = new Date().toLocaleTimeString();
  if (!area) { console.log(`[${time}] ${msg}`); return; }
  const entry = document.createElement('div');
  entry.className = 'log-entry' + (important ? ' important' : '');
  entry.textContent = `[${time}] ${msg}`;
  area.prepend(entry);
}

/* ---------------------------
   UI 初期化 / バインド
   --------------------------- */
   
// ================================
// 免責事項モーダル（既存モーダルと完全分離）
// ================================
window.addEventListener('load', () => {
  const disclaimer = document.getElementById('disclaimerModal');
  const agreeBtn = document.getElementById('agreeBtn');
  const agreedAt = localStorage.getItem('disclaimer_agreed_at');
  const now = Date.now();
  const DAY_MS = 24 * 60 * 60 * 1000; // 24時間

  // 24時間経過していない場合はスキップ
  if (agreedAt && now - parseInt(agreedAt, 10) < DAY_MS) return;

  if (disclaimer && agreeBtn) {
    // 既存の他モーダルとは別管理
    disclaimer.style.display = 'flex';
    disclaimer.setAttribute('data-active', 'true');
    document.body.style.overflow = 'hidden'; // 背景スクロール防止

    agreeBtn.addEventListener('click', () => {
      // 同意 → 記録＋閉じる
      localStorage.setItem('disclaimer_agreed_at', now.toString());
      disclaimer.style.display = 'none';
      disclaimer.removeAttribute('data-active');
      document.body.style.overflow = '';
    });
  }
});
   
window.addEventListener('DOMContentLoaded', () => {
  bindUI();
  $('eccSelect')?.dispatchEvent(new Event('change'));
  // restore theme
  const saved = localStorage.getItem('qr_theme') || 'system';
  applyTheme(saved);
  // ensure qrContainer initial small height (HTML already sets 50px; double-check)
  const qc = $('qrContainer');
  if (qc && (!qc.style.height || qc.style.height === '')) qc.style.height = '60px';
  log('アプリの初期化が完了しました。', true);
});

function bindUI() {
  // encode controls
  $('generateBtn')?.addEventListener('click', onGenerateClick);
  $('downloadZipBtn')?.addEventListener('click', onDownloadZip);
  $('clearBtn')?.addEventListener('click', onClear);
  $('showAllBtn')?.addEventListener('click', onShowAll);

  // 修正: 共通のジャンプ処理を関数としてまとめる
  const handleJump = () => {
    const v = parseInt($('jumpInput')?.value);
    // 1以上の値が入力されているか確認
    if (!isNaN(v) && v > 0) { 
      // ユーザー入力 (1, 2, 3...) を 0-based (0, 1, 2...) に変換して渡す
      // ⚠️ 修正: 分割QR番号N (1~) を配列インデックス N (1~) に変換する
      // ヘッダーはインデックス 0 なので、分割QR #N はインデックス N になります。
      // 例: 分割QR #1 -> index 1、分割QR #4 -> index 4
      jumpToIndex(v); // v-1 ではなく v を渡すことで、ヘッダーをスキップし、
                      // 分割QRの番号とインデックスを合わせます。
    } else if (v === 0) {
      // 0が入力された場合はヘッダーに移動したいという意図と仮定
      jumpToIndex(0);
    }
  };

  // modal controls
  $('nextBtn')?.addEventListener('click', () => { modalIndex = (modalIndex + 1) % generatedCount; renderModal(); });
  $('prevBtn')?.addEventListener('click', () => { modalIndex = (modalIndex - 1 + generatedCount) % generatedCount; renderModal(); });
  $('firstBtn')?.addEventListener('click', () => { modalIndex = 0; renderModal(); });
  
  // 修正: jumpBtn のイベントリスナーを handleJump に置き換え
  $('jumpBtn')?.addEventListener('click', handleJump);
  
  $('toggleSlideshow')?.addEventListener('click', () => { if (slideshowTimer) stopSlideshow(); else startSlideshow(); });
  $('modalClose')?.addEventListener('click', closeModal);

  // 修正: jumpInput の keydown イベントリスナー
  $('jumpInput')?.addEventListener('keydown', (e) => {
    // Escキーが押されたらモーダルを閉じる処理（モーダル表示中のみ）
    if (e.key === 'Escape') {
      closeModal();
      return;
    }

    // Enterキーが押されたかどうかをチェック
    if (e.key === 'Enter') {
      e.preventDefault(); // フォームのデフォルト送信を防ぐ
      handleJump(); // 共通のジャンプ処理を呼び出し
      // モーダル内でEnterで移動した場合、フォーカスを外すと便利
      $('jumpInput')?.blur(); 
    }
  });


  // decode / camera / files
  $('startCam')?.addEventListener('click', startCamera);
  $('stopCam')?.addEventListener('click', stopCamera);
  $('decodeBtn')?.addEventListener('click', onDecodeFiles);
  $('resetBtn')?.addEventListener('click', onReset);
  $('showMissingBtn')?.addEventListener('click', showMissingList);
  $('downloadRestoreBtn')?.addEventListener('click', () => verifyAndAssemble(true));

  // ✅ ログクリアボタン
  $('clearLogBtn')?.addEventListener('click', () => {
    const area = $('logArea');
    if (area) {
      area.innerHTML = '';
      log('ログをクリアしました。', true);
    }
  });

  // ✅ 初期化（ページ再読み込み）ボタン
  $('reloadBtn')?.addEventListener('click', () => {
    if (confirm('ページを再読み込みして初期状態に戻しますか？')) {
      location.reload();
    }
  });

  
// ---------------------------
// ECCレベルによって分割サイズの上限を自動調整
// ---------------------------
$('eccSelect')?.addEventListener('change', () => {
  const ecc = $('eccSelect').value;
  const chunkInput = $('chunkSize');
  if (!chunkInput) return;

  // ECCごとの理論安全上限（Base64文字数換算）
const limits = {
  L: 2900,
  M: 2280,
  Q: 1630,
  H: 1250
};

  const newMax = limits[ecc] || 2700;

  // HTML入力制御を更新
  chunkInput.max = newMax;
  // 現在値が超過していたら調整
  if (parseInt(chunkInput.value, 10) > newMax) {
    chunkInput.value = newMax;
    log(`選択したECC(${ecc})に合わせて分割上限を ${newMax} に調整しました。`, true);
  } else {
    log(`ECC(${ecc}) の推奨上限を ${newMax} に設定しました。`, true);
  }
});


/* ---------------------------
   テーマ切り替え（重複ログ防止付き）
   --------------------------- */
let suppressThemeChange = false;

function applyTheme(theme) {
  suppressThemeChange = true;
  localStorage.setItem('qr_theme', theme);

  let appliedTheme;
  if (theme === 'system') {
    appliedTheme = window.matchMedia('(prefers-color-scheme: dark)').matches
      ? 'dark'
      : 'light';
  } else {
    appliedTheme = theme;
  }

  document.documentElement.setAttribute('data-theme', appliedTheme);
  log(`テーマを変更しました： ${theme} (${appliedTheme})`, true);

  ['themeLight', 'themeDark', 'themeSystem'].forEach(id => {
    const btn = $(id);
    if (btn) {
      btn.classList.toggle('active-theme', id === `theme${theme.charAt(0).toUpperCase() + theme.slice(1)}`);
    }
  });

  setTimeout(() => { suppressThemeChange = false; }, 200);
}

// ボタンにイベントを設定
$('themeLight')?.addEventListener('click', () => applyTheme('light'));
$('themeDark')?.addEventListener('click', () => applyTheme('dark'));
$('themeSystem')?.addEventListener('click', () => applyTheme('system'));

// システムテーマ変更に追従
if (window.matchMedia) {
  const mq = window.matchMedia('(prefers-color-scheme: dark)');
  mq.addEventListener('change', () => {
    if (suppressThemeChange) return;
    const cur = localStorage.getItem('qr_theme') || 'system';
    if (cur === 'system') {
      applyTheme('system');
    }
  });
}


  // scroll loading: observe qrContainer vertical scroll
  const qc = $('qrContainer');
  if (qc) {
    qc.addEventListener('scroll', async (e) => {
      const el = e.target;
      // if near bottom (90%), load next batch
      if (!rendering && renderedIndex < generatedCount) {
        if (el.scrollTop + el.clientHeight >= el.scrollHeight * 0.9) {
          await renderNextBatch();
        }
      }
    });
  }
  
  // document全体でのキーボードイベントリスナー (以前の回答で追加)
  document.addEventListener('keydown', handleModalKeydown);
}

/* ---------------------------
   小さな待機ユーティリティ
   --------------------------- */
function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

/* ---------------------------
   カーソル操作 (wait / default)
   --------------------------- */
function setCursorWait(flag) {
  document.body.style.cursor = flag ? 'wait' : 'default';
}

/* ---------------------------
   エンコード（生成）: onGenerateClick
   - 非同期バッチで生成し、進捗ログ出力
   --------------------------- */
async function onGenerateClick() {
  const fileEl = $('fileInput');
  if (!fileEl || !fileEl.files || fileEl.files.length === 0) { alert('ファイルを選択してください'); return; }
  const file = fileEl.files[0];

// 【ファイル名の保存】
  originalFileName = file.name;

  try {
    setBusy(true);
    setCursorWait(true);
    log(`ファイルのエンコードを開始しました： ${file.name}`, true);

    // read file
    const fileBytes = new Uint8Array(await file.arrayBuffer());
    let payload = fileBytes;

	const useZip = $('compressionToggle')?.checked ?? true;
	if (useZip) {
	  log('ZIP圧縮を実行しています…');
	  const zip = new JSZip();

	  // ✅ ファイル名を ASCII セーフ化
	  const safeName = sanitizeFilename(file.name);

	  // ✅ Uint8Array に確実に変換して渡す（ArrayBufferのままだと失敗することがある）
	  const fileData = new Uint8Array(await file.arrayBuffer());

	  zip.file(safeName, fileData, { binary: true, compression: 'DEFLATE', createFolders: false, comment: '', unixPermissions: 0o644, dosPermissions: 0o644 });

	  // ✅ generateAsync でバイナリとして明示
	  payload = await zip.generateAsync({
	    type: 'uint8array',
	    compression: 'DEFLATE',
	    compressionOptions: { level: 6 },
	  });

	  log(`ZIP圧縮が完了しました。（${payload.length} bytes）`);

	  // 出力ZIP名も安全化
	  originalFileName = safeName.replace(/\.[^.]+$/, '') + '.zip';
	  var mimeType = 'application/zip';
	} else {
	  log('圧縮せずに生データを使用します。');
	  payload = new Uint8Array(await file.arrayBuffer());
	  var mimeType = file.type || 'application/octet-stream';
	}

    // hash
    log('SHA-256値を計算中…');
    const hashHex = await sha256Hex(payload);
    log('SHA-256値: ' + hashHex);
    $('restoredHash') && ($('restoredHash').textContent = hashHex);

    // base64 convert
    log('Base64形式に変換中…');
    const fullB64 = u8ToB64(payload);

    // chunk size (QRコードのサイズで分割長)
    // clamp to safe range 500..2700
    const chunkSizeInput = $('chunkSize');
    let chunkSize = parseInt(chunkSizeInput?.value || '1000', 10);
    if (!isFinite(chunkSize)) chunkSize = 1000;
    chunkSize = Math.max(10, Math.min(4000, chunkSize));
    
    // 【修正箇所: atobエラー対策】ZIP圧縮を使用しない場合、Base64デコードのためにチャンクサイズを4の倍数に調整する
    // NOTE: useZip変数は、この関数の上で既に定義され、チェックされています。
    // 🚨 前回の修正で追加された「const useZip = ...」の行は削除します。
    if (!useZip) {
      // Base64は4文字で3バイトを構成するため、atob()でデコードするチャンク長は4の倍数である必要がある。
      // 現在のチャンクサイズを4の倍数に切り捨てる。
      const oldChunkSize = chunkSize;
      chunkSize = Math.floor(chunkSize / 4) * 4;
      
      // 最小チャンクサイズが4未満にならないよう保証
      if (chunkSize < 4) chunkSize = 4;
      
      if (oldChunkSize !== chunkSize) {
          log(`非圧縮モードのため、分割サイズを ${oldChunkSize} を atob() 互換の ${chunkSize} に調整しました。`);
      }
    }
    
    chunkSizeInput.value = chunkSize; // reflect possibly clamped value

	// split into parts (Base64安全版)
	// Base64は4文字単位なので、分割位置を4の倍数に揃える
	const parts = [];
	for (let i = 0; i < fullB64.length; ) {
	  const next = Math.min(i + chunkSize, fullB64.length);
	  const end = next - (next % 4);  // ← 4の倍数に調整
	  const sliceEnd = end > i ? end : fullB64.length;  // 最後は残り全部
	  parts.push(fullB64.substring(i, sliceEnd));
	  i = sliceEnd;
	}

    // limit toggle handling
	const limitToggle = $('limitToggle');
	const maxChunks = parseInt($('maxChunksInput')?.value || '5000', 10);
	const totalChunks = parts.length; // ← 追加：この行を先に入れる

	if (limitToggle?.checked && totalChunks > maxChunks) {
	  alert(`生成予定枚数 ${totalChunks} が上限 ${maxChunks} を超えています。設定で解除するかQRサイズを大きくしてください。`);
	  setBusy(false);
	  setCursorWait(false);
	  return;
	}

	if (totalChunks > 2500) {
	  if (!confirm(`生成枚数が多いです (${totalChunks} 枚)。処理時間とメモリを多く消費します。続行しますか？`)) {
	    setBusy(false);
	    setCursorWait(false);
	    return;
	  }
	}


    // header
	const headerObj = {
	  v: 9,
	  name: file.name,
	  mime: file.type || 'application/octet-stream',
	  totalChunks: parts.length, // ← 修正: 正しい分割数を設定
	  hash: hashHex,
	  compression: useZip ? 'zip' : 'none',
	  createdAt: new Date().toISOString()
	};

    const headerB64 = strToB64Utf8(JSON.stringify(headerObj));

    // build qItems: FILEHDR + CHUNK lines
    qItems = [];
    qItems.push('FILEHDR|' + headerB64);

    // Build chunk lines but do not render now; also compute CRC per chunk
    // We'll push in qItems but generate canvases lazily to avoid large memory upfront
    for (let i = 0; i < parts.length; i++) {
      const payloadPart = parts[i];
      const u8 = b64ToU8(payloadPart);
      const crc = CRC32(u8);
      qItems.push(`CHUNK|${i + 1}/${parts.length}|${payloadPart}|${crc}`);
      // occasionally yield to keep UI responsive while building array for very large files
      if (i % 200 === 0) await sleep(1);
      // progress log periodically
      const now = Date.now();
      if (now - lastProgressLogAt > PROGRESS_LOG_INTERVAL_MS) {
        lastProgressLogAt = now;
        log(`QRデータを構築中: ${i + 1}/${parts.length} 分割  （${qItems.length} 行）`);
      }
    }

    generatedCount = qItems.length;
    renderedIndex = 0;

	// (修正) 生成後は固定高さを与えてスクロール領域を確保する
	const qContainer = $('qrContainer');
	if (qContainer) {
	  // 生成前は 50px。生成後は一覧をスクロールで見られる高さに固定する
	  qContainer.style.height = '280px';      // ← 表示高さ（見た目を変えたくないなら適宜調整可）
	  qContainer.style.overflowY = 'auto';    // 縦スクロールを確実に有効化
	  qContainer.scrollTop = 0;               // 表示位置を先頭に戻す
	}

    // clear old thumbnails and start drawing initial batch
    const qrRow = $('qrRow');
    if (qrRow) qrRow.innerHTML = '';

    log(`QRコードの生成を開始します（初回 ${INITIAL_DISPLAY} 枚）。`, true);
    await renderNextBatch(); // draw initial set (non-blocking)

  // final UI state
  $('downloadZipBtn') && ($('downloadZipBtn').disabled = false);
  // ✅ 生成が正常に完了したときだけ「全て生成」ボタンを有効化
  $('showAllBtn') && ($('showAllBtn').disabled = false);
  log(`QRコードの生成が完了しました：${generatedCount} 枚（ヘッダーQRを除く分割QR：${parts.length} 枚）`, true);
  } catch (e) {
    console.error(e);
    alert('生成中にエラーが発生しました: ' + (e.message || e));
    log('生成エラー: ' + (e.message || e), true);
  } finally {
    setBusy(false);
    setCursorWait(false);
  }
}

/*
   描画バッチ / モーダル / ZIP生成
*/

/* ---------------------------
   描画バッチ: renderNextBatch
   - DRAW_BATCH 単位で非同期描画し、UIフリーズを回避
   - 描画ごとに小さな sleep を入れてブラウザに余裕を与える
   --------------------------- */
async function renderNextBatch() {
  if (rendering) return;
  rendering = true;
  const qrRow = $('qrRow');
  if (!qrRow) { rendering = false; return; }

  const ecc = $('eccSelect')?.value || 'M';
  const batch = DRAW_BATCH;
  const start = renderedIndex;
  const end = Math.min(renderedIndex + batch, generatedCount);

  log(`QR生成処理を開始: ${start + 1} ～ ${end} / ${generatedCount}`);

  for (let i = start; i < end; i++) {
    const frag = qItems[i];
    // create card
    const fragDiv = document.createElement('div');
    fragDiv.className = 'qr-frag';
    const cap = document.createElement('div');
    cap.className = 'caption';
	    if (i === 0) {
		  cap.textContent = 'ヘッダーQR';
		} else {
		  cap.textContent = `分割QR #${i}/${generatedCount - 1}`;
		}
    fragDiv.appendChild(cap);
    const canvas = document.createElement('canvas');
    fragDiv.appendChild(canvas);
    qrRow.appendChild(fragDiv);

    // draw progressively using requestAnimationFrame for smoothness
    await new Promise(resolve => requestAnimationFrame(async () => {
      try {
        // Use moderate thumbnail size for list to save memory
        new QRious({ element: canvas, value: frag, level: ecc, size: 220 });
        // attach click to open modal at index
        fragDiv.addEventListener('click', () => openModal(i));
      } catch (e) {
        console.error('QR生成エラー', e);
      } finally {
        resolve();
      }
    }));

    // small yield to avoid long blocking of main thread when many items
    if ((i - start) % 5 === 0) await sleep(BATCH_SLEEP_MS);
    // periodic progress logging
    const now = Date.now();
    if (now - lastProgressLogAt > PROGRESS_LOG_INTERVAL_MS) {
      lastProgressLogAt = now;
      log(`QRコード生成中： ${i + 1} / ${generatedCount}`);
    }
  }

  renderedIndex = end;
  rendering = false;

  log(`QR生成処理が完了： ${renderedIndex} / ${generatedCount}`);

  // if renderedIndex < generatedCount, ensure a visible scrollbar (qrContainer has overflow-y:auto)
  const qc = $('qrContainer');
  if (qc) {
    // if content height is less than container and more items exist, expand container slightly
    // otherwise keep as-is (auto after generation)
    // no-op here — css controls scrollbar
  }
}

/* ---------------------------
   showAll convenience (user-confirmed heavy op)
   --------------------------- */
async function onShowAll() {
  if (!confirm('すべて表示すると大量のメモリを消費します。続行しますか？')) return;
  while (renderedIndex < generatedCount) {
    await renderNextBatch();
    // tiny pause
    await sleep(8);
  }
  log('すべてのQRコードの表示が完了しました。', true);
}

/* ---------------------------
   ZIP作成: onDownloadZip
   - すべてのQRをPNGとしてJSZipに追加する
   - 進捗ログを逐次表示
   --------------------------- */
async function onDownloadZip() {
  if (!generatedCount || !qItems.length) { alert('生成されたQRがありません'); return; }
  setBusy(true);
  setCursorWait(true);
  try {
    log('QR画像のZIP作成を開始します…', true);
    const zip = new JSZip();
    for (let i = 0; i < generatedCount; i++) {
      // try to reuse rendered canvas if exists
      let canvas = null;
      const fragDiv = $('qrRow')?.children[i];
      if (fragDiv) canvas = fragDiv.querySelector('canvas');
      if (!canvas) {
        // offscreen canvas
        canvas = document.createElement('canvas');
        try {
          new QRious({ element: canvas, value: qItems[i], level: $('eccSelect')?.value || 'M', size: 600 });
        } catch (e) { console.error('オフスクリーンQR生成失敗', e); }
      }
      // convert to blob
      const blob = await new Promise(res => canvas.toBlob(res, 'image/png'));
      zip.file(`qr_${String(i).padStart(5, '0')}.png`, blob);
      if (i % 20 === 0) await sleep(4); // yield
      if (Date.now() - lastProgressLogAt > PROGRESS_LOG_INTERVAL_MS) {
        lastProgressLogAt = Date.now();
        log(`ZIPに追加中： ${i + 1} / ${generatedCount}`);
      }
    }
    const outBlob = await zip.generateAsync({ type: 'blob' }, (meta) => {
      // meta.percent can be used for progress if desired
      if (Date.now() - lastProgressLogAt > PROGRESS_LOG_INTERVAL_MS) {
        lastProgressLogAt = Date.now();
        log(`ZIP生成中： ${Math.round(meta.percent)}%`);
      }
    });
    
    // 【修正箇所】ダウンロードファイル名を変更
    let downloadName = 'qr_bundle.zip';
    if (originalFileName) {
        // 元のファイル名に "_qr.zip" を付加する (例: file.txt -> file.txt_qr.zip)
        downloadName = `${originalFileName}_qr.zip`;
    }
    
    saveAs(outBlob, downloadName);
    log('ZIPの作成が完了しました。', true);
  } catch (e) {
    console.error(e);
    alert('ZIP作成中にエラーが発生しました: ' + (e.message || e));
    log('ZIP作成エラー: ' + (e.message || e), true);
  } finally {
    setBusy(false);
    setCursorWait(false);
  }
}
/* ---------------------------
   モーダル: openModal / renderModal / closeModal
   - canvas の描画サイズをウィンドウに合わせて動的に算出
   --------------------------- */
function openModal(index) {
  if (!qItems || qItems.length === 0) return;
  modalIndex = index;
  renderModal();
  const modal = $('modal');
  if (modal) modal.classList.add('show');
  // prevent body scroll while modal open
  document.body.style.overflow = 'hidden';
}

function closeModal() {
  const modal = $('modal');
  if (modal) modal.classList.remove('show');
  document.body.style.overflow = '';
  stopSlideshow();
}

function renderModal() {
  const canvas = $('modalCanvas');
  if (!canvas) return;
  
  $('modal')?.classList.add('show');
  
  // 1. 最大サイズをビューポートに基づいて決定
  const maxW = Math.floor(window.innerWidth * 0.98);
  const maxH = Math.floor(window.innerHeight * 0.95);
  
  // 2. QRiousに渡すサイズを決定
  //    ここでQRコードの最大サイズを制限します。
  //    (maxW, maxH, 2000) を (maxW, maxH, 600) に変更
  //    これにより、キャンバスの内部解像度が 600x600 を超えなくなります。
  const MAX_QR_SIZE = 2000; // 👈 ここで最大サイズを 600px に設定
  const size = Math.min(maxW, maxH, MAX_QR_SIZE); // 2000 を MAX_QR_SIZE に変更

  // use QRious to draw to canvas
  try {
    const qRiousValue = qItems[modalIndex]?.data || qItems[modalIndex];
    new QRious({ 
      element: canvas, 
      value: qRiousValue,
      level: $('eccSelect')?.value || 'M', 
      size: size // 計算されたサイズを渡す
    });
  } catch (e) {
    console.error('modal QR 描画エラー', e);
  }
  
  // 3. キャンバスのCSSスタイル
  canvas.style.maxWidth = '85%';
  canvas.style.maxHeight = '85%'; 
  
// ⚠️ 修正箇所: モーダルステータスのテキストを修正
  const currentNumber = modalIndex + 1;
  let label = '';

  if (modalIndex === 0) {
    // 最初の1枚目（インデックス0）はヘッダーとして表示
    label = `ヘッダーQR`;
  } else {
    // 2枚目以降（インデックス1から）は分割QRとして表示
    const chunkNumber = modalIndex; // 分割QRの番号はインデックスに等しい
    label = `分割QR #${chunkNumber}`; 
  }

  // ステータスを更新
  $('modalStatus') && ($('modalStatus').textContent = `${label} / ${generatedCount - 1}  (全体 ${currentNumber} 枚目)`);
}

/* ---------------------------
   スライドショー (modal 内)
   --------------------------- */
function startSlideshow() {
  const interval = parseFloat($('slideInterval')?.value || '1.00');
  if (!interval || interval <= 0) { alert('正しい切替間隔を入力してください'); return; }
  if (slideshowTimer) clearInterval(slideshowTimer);
  slideshowTimer = setInterval(() => {
    modalIndex = (modalIndex + 1) % generatedCount;
    renderModal();
  }, Math.max(10, Math.round(interval * 1000)));
  $('toggleSlideshow') && ($('toggleSlideshow').textContent = '連続表示停止');
}

function stopSlideshow() {
  if (slideshowTimer) clearInterval(slideshowTimer);
  slideshowTimer = null;
  $('toggleSlideshow') && ($('toggleSlideshow').textContent = '連続表示開始');
}

/* ---------------------------
   モーダル: キーボード操作の処理
   --------------------------- */
function handleModalKeydown(e) {
  // モーダルが表示されていない場合は何もしない
  if (!$('modal')?.classList.contains('show')) {
    return;
  }

  // スライドショー中はキー操作を無効化
  if (slideshowTimer !== null) {
    return;
  }
  
  // 入力フィールドがアクティブな場合は、キー入力を妨げない
  if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
    return;
  }

  // generatedCount が 0 以下、または 1 枚のみの場合はスキップ
  if (generatedCount <= 1) return;


  switch (e.key) {
    case 'ArrowLeft': // 左矢印キー (←)
      e.preventDefault(); // スクロールなどブラウザのデフォルト動作を抑止
      
      // 修正: 0 から -1 になった場合、最後のインデックス (generatedCount - 1) にループ
      modalIndex = (modalIndex - 1 + generatedCount) % generatedCount;
      renderModal();
      break;
      
    case 'ArrowRight': // 右矢印キー (→)
      e.preventDefault();
      
      // 修正: generatedCount に到達した場合、0 (ヘッダー) にループ
      modalIndex = (modalIndex + 1) % generatedCount;
      renderModal();
      break;
      
    case 'Escape': // Escキー
      e.preventDefault();
      closeModal();
      break;
  }
}

/* ---------------------------
   QR 拡大（改良版：縦スクロールで先頭へ戻れる / 短いときは縦中央）
   --------------------------- */
window.addEventListener('DOMContentLoaded', () => {
  const qrContainer = document.getElementById('qrContainer');
  if (!qrContainer) return;

  // 重複防止
  qrContainer.querySelector('#qrExpandBtn')?.remove();
  qrContainer.querySelector('.close-btn')?.remove();

	// 拡大ボタン作成（← qrContainer 内に配置）
	const expandBtn = document.createElement('button');
	expandBtn.id = 'qrExpandBtn';
	expandBtn.textContent = '🔍';
	expandBtn.title = 'QR一覧を全画面表示';
	qrContainer.appendChild(expandBtn); // ← これが重要（bodyではなくcontainer内）

	// ×ボタンは container 内でもよいが fixed なのでどこでもOK
	const closeBtn = document.createElement('button');
	closeBtn.className = 'close-btn';
	closeBtn.innerHTML = '×';
	closeBtn.style.display = 'none';
	qrContainer.appendChild(closeBtn);

  // ラップ状態を記録するための参照
  let innerWrapper = null;

  const wrapChildren = () => {
    if (innerWrapper) return;
    // 作る wrapper
    innerWrapper = document.createElement('div');
    innerWrapper.className = 'qr-inner';
    // move all child nodes except expandBtn and closeBtn into wrapper
    const nodes = Array.from(qrContainer.childNodes);
    nodes.forEach(node => {
      if (node === expandBtn || node === closeBtn) return;
      innerWrapper.appendChild(node);
    });
    qrContainer.appendChild(innerWrapper);
  };

  const unwrapChildren = () => {
    if (!innerWrapper) return;
    // move children back to qrContainer before removing wrapper
    const nodes = Array.from(innerWrapper.childNodes);
    nodes.forEach(node => qrContainer.appendChild(node));
    innerWrapper.remove();
    innerWrapper = null;
  };

  const openFullscreen = async () => { // ← async を追加
    wrapChildren();
    // small delay to ensure wrapper is in DOM (not strictly required)
    requestAnimationFrame(async () => {
      qrContainer.classList.add('fullscreen');
      document.body.style.overflow = 'hidden';
      closeBtn.style.display = 'block';
      expandBtn.style.display = 'none';
      // scroll to top of wrapper so user sees first items
      if (innerWrapper) innerWrapper.scrollIntoView({ block: 'start' });

      // 💡 追加部分：拡大時に未描画のQRを強制的に描画
      if (!rendering && renderedIndex < generatedCount) {
        await renderNextBatch();
      }
    });
  };

  const closeFullscreen = () => {
    qrContainer.classList.remove('fullscreen');
    document.body.style.overflow = '';
    closeBtn.style.display = 'none';
    expandBtn.style.display = 'block';
    // unwrap after a tick so any CSS transition won't disrupt DOM
    setTimeout(() => {
      unwrapChildren();
    }, 0);
  };

  // イベント
  expandBtn.addEventListener('click', e => {
    e.stopPropagation();
    openFullscreen();
  });
  closeBtn.addEventListener('click', e => {
    e.stopPropagation();
    closeFullscreen();
  });

  // 背景クリック（フルスクリーン中のみ閉じる）
  qrContainer.addEventListener('click', e => {
    if (e.target === qrContainer && qrContainer.classList.contains('fullscreen')) {
      closeFullscreen();
    }
  });

  // QR要素クリック時：フルスクリーン中なら先に閉じる（既存モーダル処理のため）
  qrContainer.addEventListener('click', e => {
    const target = e.target;
    const isQrElement =
      target.tagName === 'CANVAS' ||
      target.tagName === 'IMG' ||
      (target.classList && target.classList.contains('qr-frag'));

    if (isQrElement && qrContainer.classList.contains('fullscreen')) {
      // まず拡大解除してから既存のQRクリック処理を継続させる
      // 解除を同期で行うと既存のクリックハンドラが続くので自然にモーダルが開きます
      closeFullscreen();
      // 注意: 必要ならタイミング調整（setTimeout(..., 50)）を入れてもよい
    }
  });

  // ESC キーで閉じる
  document.addEventListener('keydown', e => {
    if (e.key === 'Escape' && qrContainer.classList.contains('fullscreen')) {
      closeFullscreen();
    }
  });
});

/* ---------------------------
   ユーティリティ: setBusy
   --------------------------- */
function setBusy(flag) {
  $('generateBtn') && ($('generateBtn').disabled = flag);
  $('downloadZipBtn') && ($('downloadZipBtn').disabled = flag || (generatedCount === 0));
  $('clearBtn') && ($('clearBtn').disabled = flag);
  $('decodeBtn') && ($('decodeBtn').disabled = flag);
}

/* カメラ / デコード / 復元 / ユーティリティ 等
*/

/* ---------------------------
   カメラ: start / stop / tick
   --------------------------- */
async function startCamera() {
  try {
    if (cameraStream) return;
    cameraStream = await navigator.mediaDevices.getUserMedia({ video: { facingMode: 'environment' }, audio: false });
    cameraVideoEl = $('cameraPreview');
    if (!cameraVideoEl) {
      cameraVideoEl = document.createElement('video');
      cameraVideoEl.setAttribute('playsinline', '');
      cameraVideoEl.autoplay = true;
      cameraVideoEl.muted = true;
      document.body.appendChild(cameraVideoEl);
    }
    cameraVideoEl.srcObject = cameraStream;
    cameraVideoEl.style.display = 'block';
    await cameraVideoEl.play();
    $('startCam') && ($('startCam').disabled = true);
    $('stopCam') && ($('stopCam').disabled = false);
    restored.isRunning = true;
    log('カメラを起動しました。', true);
    cameraTick();
  } catch (e) {
    console.error('startCamera error', e);
    alert('カメラの起動に失敗しました: ' + (e.message || e));
    log('カメラ起動失敗: ' + (e.message || e), true);
  }
}

function stopCamera() {
  if (cameraStream) {
    cameraStream.getTracks().forEach(t => t.stop());
    cameraStream = null;
  }
  if (cameraVideoEl) {
    try { cameraVideoEl.pause(); cameraVideoEl.srcObject = null; } catch (e) {}
    cameraVideoEl.style.display = 'none';
  }
  if (cameraAnim) cancelAnimationFrame(cameraAnim);
  cameraAnim = null;
  restored.isRunning = false;
  $('startCam') && ($('startCam').disabled = false);
  $('stopCam') && ($('stopCam').disabled = true);
  log('カメラを停止しました。', true);
}

function cameraTick() {
  try {
    if (!cameraVideoEl || cameraVideoEl.readyState !== cameraVideoEl.HAVE_ENOUGH_DATA) {
      cameraAnim = requestAnimationFrame(cameraTick);
      return;
    }
    const w = cameraVideoEl.videoWidth;
    const h = cameraVideoEl.videoHeight;
    if (!w || !h) { cameraAnim = requestAnimationFrame(cameraTick); return; }

    const c = document.createElement('canvas');
    c.width = w; c.height = h;
    const ctx = c.getContext('2d');
    ctx.drawImage(cameraVideoEl, 0, 0, w, h);
    try {
      const imgd = ctx.getImageData(0, 0, w, h);
      const code = jsQR(imgd.data, imgd.width, imgd.height, { inversionAttempts: 'attemptBoth' });
      if (code && code.data) handleDecodedString(code.data);
    } catch (e) {
      // getImageData may throw on cross-origin streams in rare cases; ignore
      console.warn('cameraTick getImageData error', e);
    }
  } catch (e) {
    console.error('cameraTick top error', e);
  } finally {
    cameraAnim = requestAnimationFrame(cameraTick);
  }
}

/* ---------------------------
   画像/ZIP デコード (ファイル入力)
   --------------------------- */
async function onDecodeFiles() {
  const input = $('imageFiles');
  if (!input || !input.files || input.files.length === 0) { alert('画像またはZIPを選択してください'); return; }
  const files = Array.from(input.files);
  log(`ファイルのデコードを開始しました： ${files.length} 件`, true);
  setBusy(true);
  try {
    for (const f of files) {
      if (f.name.toLowerCase().endsWith('.zip')) {
        try {
          const z = await JSZip.loadAsync(f);
          const names = Object.keys(z.files);
          for (const n of names) {
            const entry = z.files[n];
            if (!entry.dir && /\.(png|jpe?g|bmp|gif)$/i.test(n)) {
              const blob = await entry.async('blob');
              await decodeImageBlob(blob);
            }
          }
        } catch (e) {
          log('ZIP展開エラー: ' + (e.message || e), true);
        }
      } else {
        await decodeImageFile(f);
      }
      // small yield between files
      await sleep(6);
    }
    log('ファイルのデコードが完了しました。', true);
  } finally {
    setBusy(false);
  }
}

function decodeImageFile(file) {
  return new Promise((resolve) => {
    const fr = new FileReader();
    fr.onload = (ev) => {
      const img = new Image();
      img.onload = async () => { await decodeImageElement(img); resolve(); };
      img.onerror = () => { log('画像読み込み失敗: ' + file.name, true); resolve(); };
      img.src = ev.target.result;
    };
    fr.readAsDataURL(file);
  });
}
function decodeImageBlob(blob) {
  return new Promise((resolve) => {
    const url = URL.createObjectURL(blob);
    const img = new Image();
    img.onload = async () => { await decodeImageElement(img); URL.revokeObjectURL(url); resolve(); };
    img.onerror = () => { URL.revokeObjectURL(url); log('ZIP内の画像の読み込みに失敗しました。', true); resolve(); };
    img.src = url;
  });
}
async function decodeImageElement(img) {
  try {
    const c = document.createElement('canvas');
    c.width = img.width; c.height = img.height;
    const ctx = c.getContext('2d');
    ctx.drawImage(img, 0, 0);
    const imgd = ctx.getImageData(0, 0, c.width, c.height);
    const code = jsQR(imgd.data, imgd.width, imgd.height, { inversionAttempts: 'attemptBoth' });
    if (code && code.data) handleDecodedString(code.data);
    else log('画像内にQRなし', false);
  } catch (e) {
    log('画像デコードエラー: ' + (e.message || e), true);
  }
}

/* ---------------------------
   受信文字列の処理（HEADER / CHUNK）
   --------------------------- */
function handleDecodedString(text) {
  if (!text || typeof text !== 'string') return;
  const now = Date.now();
  if (recentSeen.has(text) && (now - recentSeen.get(text) < 800)) return; // debounce
  recentSeen.set(text, now);
  // cleanup occasionally
  if (recentSeen.size > 2000) {
    for (const [k, t] of recentSeen.entries()) if (now - t > 5000) recentSeen.delete(k);
  }

  try {
    if (text.startsWith('FILEHDR|')) {
      const b64 = text.split('|')[1];
      const json = JSON.parse(b64ToStrUtf8(b64));

      // 🚑 totalChunks を安全に数値化
      const chunks = json.totalChunks;
      restored.totalChunks =
        typeof chunks === 'number'
          ? chunks
          : (typeof chunks === 'object' && 'n' in chunks ? chunks.n : 0);

      restored.name = json.name || '';
      restored.mime = json.mime || 'application/octet-stream';
      restored.hash = json.hash || '';

      $('totalChunks') && ($('totalChunks').textContent = restored.totalChunks);
      $('restoredName') && ($('restoredName').textContent = restored.name);
      $('restoredHash') && ($('restoredHash').textContent = restored.hash);

      log(`ヘッダーを読み取りました： ${restored.name} 全${restored.totalChunks}コード`, true);
    } else if (text.startsWith('CHUNK|')) {
      const parts = text.split('|');
      if (parts.length < 3) return;
      const meta = parts[1];
      const [nStr, tStr] = meta.split('/');
      const n = parseInt(nStr, 10);
      const t = parseInt(tStr, 10);
      const payload = parts[2];
      const crcPart = parts[3] ? parts[3] : null;

      if (!restored.totalChunks && t) {
        restored.totalChunks = t;
        $('totalChunks') && ($('totalChunks').textContent = restored.totalChunks);
      }

      if (restored.fragments[String(n)]) {
        log(`分割QR #${n} が既に存在します。（重複）`, false);
        updateProgress(false);
        return;
      }

	if (crcPart) {
	  const u8 = b64ToU8(payload);
	  const crcCalc = CRC32(u8);
	  const crcGot = parseInt(crcPart, 10);
	  const ok = crcCalc === crcGot;
	  const crcHex = crcCalc.toString(16).toUpperCase().padStart(8, '0'); // 8桁の16進表示
	  log(`CRC32: #${n} (${crcHex}) → ${ok ? '一致' : '不一致 (破棄)'}`, !ok);
	  if (!ok) return;
	}


      restored.fragments[String(n)] = payload;
      restored.received = Object.keys(restored.fragments).length;
      $('receivedChunks') && ($('receivedChunks').textContent = restored.received);
      updateProgress(false);
      log(`分割QR #${n} 読取成功 (${restored.received}/${restored.totalChunks})`, false);

      if (restored.totalChunks > 0 && restored.received === restored.totalChunks) {
        log('すべてのQRコードを読み取りました。復元を開始します。', true);
        updateProgress(true);
      }
    } else {
      log('不明データ受信（無視）', false);
    }
  } catch (e) {
    log('デコード処理例外: ' + (e.message || e), true);
  }
}

/* ---------------------------
   進捗表示 / 欠損率更新
   --------------------------- */
function updateProgress(final) {
  const rec = restored.received;
  const tot = restored.totalChunks || 0;
  const pct = tot > 0 ? Math.round((rec / tot) * 100) : 0;
  const prog = $('progressBar');
  if (prog) { prog.style.width = pct + '%'; prog.textContent = `${rec}/${tot} (${pct}%)`; }
  const status = $('statusText');
  if (status) {
    const loss = tot > 0 ? ((1 - (rec / tot)) * 100).toFixed(1) : '—';
    status.textContent = tot > 0 ? `読み取り: ${rec}/${tot}` : '状態: ヘッダー待ち';
    $('lossRate') && ($('lossRate').textContent = `${loss}%`);
  }
  $('showMissingBtn') && ($('showMissingBtn').disabled = !(tot > 0));
  if (tot > 0 && rec === tot) {
    $('downloadRestoreBtn') && ($('downloadRestoreBtn').disabled = false);
    if (final) verifyAndAssemble();
  } else {
    $('downloadRestoreBtn') && ($('downloadRestoreBtn').disabled = true);
  }
}

/* ---------------------------
   欠損リスト表示
   --------------------------- */
function showMissingList() {
  if (!restored.totalChunks) { alert('ヘッダーが読み込まれていません'); return; }
  const missing = [];
  for (let i = 1; i <= restored.totalChunks; i++) if (!restored.fragments[String(i)]) missing.push(i);
  const preview = missing.slice(0, 500).join(', ');
  alert(missing.length === 0 ? '✅ 全ての読取に成功しています。（不足コードなし）' : `不足分(${missing.length}):\n${preview}${missing.length>500?'\n...and more':''}`);
}

/* ---------------------------
   検証・結合・ダウンロード（ZIP自動解凍版・安定版）
   --------------------------- */
async function verifyAndAssemble(promptIfIncomplete = false) {
  if (!restored.totalChunks) {
    alert('ヘッダが読み込まれていません');
    return;
  }

  if (Object.keys(restored.fragments).length !== restored.totalChunks) {
    if (promptIfIncomplete) {
      alert('まだすべての分割QRが揃っていません');
      return;
    }
  }

  // チャンクを結合
  let fullB64 = '';
  for (let i = 1; i <= restored.totalChunks; i++) {
    const chunk = restored.fragments[String(i)];
    if (!chunk) {
      alert('不足しているQRコードがあります: #' + i);
      return;
    }
    fullB64 += chunk;
  }

  let u8;
  try {
    // Base64チェック
    if (/^[A-Za-z0-9+/=]+$/.test(fullB64)) {
      u8 = b64ToU8(fullB64);
    } else {
      console.warn('Base64ではないデータを検出: ZIPの可能性あり');
      u8 = new TextEncoder().encode(fullB64);
    }
  } catch (e) {
    console.error('Base64デコード失敗。ZIPとして処理します:', e);
    u8 = new TextEncoder().encode(fullB64);
  }

  // ZIPファイルを検出
  const isZip =
    u8[0] === 0x50 && u8[1] === 0x4b && u8[2] === 0x03 && u8[3] === 0x04;
  if (isZip) {
    restored.mime = 'application/zip';
    if (!restored.name.toLowerCase().endsWith('.zip')) {
      restored.name = restored.name.replace(/\.[^.]+$/, '') + '.zip';
    }
  }

  // ハッシュ検証
  const computed = await sha256Hex(u8);
  if (restored.hash && restored.hash !== computed) {
    if (
      !confirm(
        `ハッシュSHA-256値 不一致（破損の可能性）\nヘッダ: ${restored.hash}\n計算: ${computed}\n続行しますか？`
      )
    )
      return;
  } else if (restored.hash) {
    log('ハッシュSHA-256値の計算結果:  一致');
  }

  $('restoredHash') && ($('restoredHash').textContent = computed);

  let blob, name;

  /* ---------------------------
     ZIP自動展開処理
  --------------------------- */
  if (isZip) {
    try {
      log('ZIPファイルを展開しています…');

      // ZIP終端シグネチャ(0x50,0x4B,0x05,0x06)を検索して末尾補正
      const endSig = [0x50, 0x4b, 0x05, 0x06];
      const sigIndex = (() => {
        for (let i = u8.length - 4; i >= 0; i--) {
          if (
            u8[i] === endSig[0] &&
            u8[i + 1] === endSig[1] &&
            u8[i + 2] === endSig[2] &&
            u8[i + 3] === endSig[3]
          ) {
            return i;
          }
        }
        return -1;
      })();

      const fixedU8 = sigIndex > 0 ? u8.slice(0, sigIndex + 22) : u8;

      const zip = await JSZip.loadAsync(fixedU8);
      const files = Object.keys(zip.files);
      log(`ZIP内のファイルを検出しました：${files.length} 件`);

      if (files.length === 1) {
        const entry = zip.files[files[0]];
        const innerData = await entry.async('uint8array');
        blob = new Blob([innerData], { type: 'application/octet-stream' });
        name = sanitizeFilename(entry.name);
        log(`ZIP内の1件のファイルを復元しました：${name}`, true);
      } else {
        blob = new Blob([fixedU8], { type: 'application/zip' });
        name = sanitizeFilename(restored.name);
        log(`ZIP内に${files.length}件のファイルが検出されました。ZIPのまま保存します。`,
          true
        );
      }
    } catch (e) {
      console.error('ZIP展開に失敗。ZIPのまま保存します:', e);
      blob = new Blob([u8], { type: 'application/zip' });
      name = sanitizeFilename(restored.name);
    }
  } else {
    // 通常ファイル
    blob = new Blob([u8], {
      type: restored.mime || 'application/octet-stream',
    });
    name = sanitizeFilename(restored.name || `restored_${Date.now()}`);
  }

  /* ---------------------------
     保存処理
  --------------------------- */
  try {
    saveAs(blob, name);
    log('復元ファイルのダウンロード: ' + name, true);
  } catch (e) {
    console.error('saveAsに失敗:', e);
    alert('ファイル保存に失敗しました: ' + e.message);
  }
}

/* ---------------------------
   リセット / クリア
   --------------------------- */
function onReset() {
  restored = { fragments: {}, totalChunks: 0, name: '', mime: '', hash: '', received: 0, isRunning: false };
  recentSeen.clear();
  $('totalChunks') && ($('totalChunks').textContent = '-');
  $('receivedChunks') && ($('receivedChunks').textContent = '0');
  $('restoredName') && ($('restoredName').textContent = '-');
  $('restoredHash') && ($('restoredHash').textContent = '-');
  $('progressBar') && (($('progressBar').style.width = '0%'), ($('progressBar').textContent = '0%'));
  $('statusText') && ($('statusText').textContent = '状態: リセット済み');
  $('downloadRestoreBtn') && ($('downloadRestoreBtn').disabled = true);
  $('showMissingBtn') && ($('showMissingBtn').disabled = true);
  log('復元状態をリセットしました。', true);
}

function onClear() {
  qItems = [];
  generatedCount = 0;
  renderedIndex = 0;
  const qrRow = $('qrRow');
  if (qrRow) qrRow.innerHTML = '';
  const qContainer = $('qrContainer');
  if (qContainer) qContainer.style.height = '60px';
  $('downloadZipBtn') && ($('downloadZipBtn').disabled = true);
  log('生成データをクリアしました。', true);
}

/* ---------------------------
   テーマ適用
   --------------------------- */
function applyTheme(mode) {
  const body = document.body;
  if (!body) return;

  // --- テーマ適用 ---
  if (mode === 'light') {
    body.setAttribute('data-theme', 'light');
  } else if (mode === 'dark') {
    body.setAttribute('data-theme', 'dark');
  } else {
    // システム設定を検出して適用
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    body.setAttribute('data-theme', prefersDark ? 'dark' : 'light');

    // OSテーマ変化を自動追従
    const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');
    const listener = (e) => {
      body.setAttribute('data-theme', e.matches ? 'dark' : 'light');
      updateThemeButtonHighlight('system');
    };
    mediaQuery.removeEventListener('change', listener);
    mediaQuery.addEventListener('change', listener);
  }

  // --- ボタンの強調状態を更新 ---
  updateThemeButtonHighlight(mode);

  log('テーマ適用: ' + mode, false);
}

/* ボタン強調表示（ライト／ダーク／システム） */
function updateThemeButtonHighlight(currentMode) {
  const map = {
    light: document.getElementById('themeLight'),
    dark: document.getElementById('themeDark'),
    system: document.getElementById('themeSystem')
  };
  Object.keys(map).forEach(key => {
    const btn = map[key];
    if (!btn) return;
    if (key === currentMode) {
      btn.classList.add('active-theme');
    } else {
      btn.classList.remove('active-theme');
    }
  });
}

/* ページ読み込み時の初期化（イベント紐付け） */
window.addEventListener('DOMContentLoaded', () => {
  const btnLight = document.getElementById('themeLight');
  const btnDark = document.getElementById('themeDark');
  const btnSys = document.getElementById('themeSystem');

  if (btnLight) btnLight.addEventListener('click', () => applyTheme('light'));
  if (btnDark) btnDark.addEventListener('click', () => applyTheme('dark'));
  if (btnSys) btnSys.addEventListener('click', () => applyTheme('system'));
});



/* ---------------------------
   小さなユーティリティ（エクスポート）
   --------------------------- */
window.qrApp = {
  onGenerateClick, onDownloadZip, onClear, onShowAll,
  startCamera, stopCamera, onDecodeFiles, onReset,
  showMissingList, verifyAndAssemble, applyTheme
};


/* ---------------------------
   モーダル: jumpToIndex
   --------------------------- */
function jumpToIndex(index) {
  // indexは0ベースである必要があります (例: ヘッダーは 0)
  if (index >= 0 && index < generatedCount) {
    // 渡されたインデックスをそのまま使用
    modalIndex = index;
    renderModal(); 
  } else {
    // 範囲外の値が入力された場合の処理
    alert(`無効なQRコード番号です。1 から ${generatedCount} の間の値を入力してください。`);
  }
}